<?php
/**
 * Hero Slider Image Upload - Verification Test
 * Test if image upload functionality is working correctly
 */

require_once 'config.php';
session_start();

// Check authentication
$isLoggedIn = isset($_SESSION['user_id']);
$isAdmin = $isLoggedIn && $_SESSION['user_type'] === 'admin';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hero Slider Image Upload - Test</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #109c3d 0%, #0f7938 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .content {
            padding: 30px;
        }
        
        .test-section {
            margin-bottom: 30px;
            padding: 20px;
            border-left: 4px solid #109c3d;
            background: #f8fbff;
            border-radius: 8px;
        }
        
        .test-section h2 {
            color: #0f172a;
            font-size: 18px;
            margin-bottom: 15px;
        }
        
        .test-item {
            display: flex;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid #ddd;
        }
        
        .test-item:last-child {
            border-bottom: none;
        }
        
        .status-icon {
            font-size: 20px;
            margin-right: 15px;
            width: 30px;
            text-align: center;
        }
        
        .status-icon.success {
            color: #10b981;
        }
        
        .status-icon.error {
            color: #ef4444;
        }
        
        .status-icon.warning {
            color: #f59e0b;
        }
        
        .status-text {
            flex: 1;
        }
        
        .status-text strong {
            color: #0f172a;
            display: block;
            margin-bottom: 3px;
        }
        
        .status-text small {
            color: #64748b;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
        }
        
        .alert-danger {
            background: #fee2e2;
            border: 1px solid #fca5a5;
            color: #991b1b;
        }
        
        .alert-success {
            background: #d1fae5;
            border: 1px solid #6ee7b7;
            color: #065f46;
        }
        
        .alert-warning {
            background: #fef3c7;
            border: 1px solid #fcd34d;
            color: #92400e;
        }
        
        .alert i {
            margin-right: 10px;
            font-size: 18px;
        }
        
        .code-block {
            background: #0f172a;
            color: #22d3ee;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            overflow-x: auto;
            margin-top: 10px;
        }
        
        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: #109c3d;
            color: white;
        }
        
        .btn-primary:hover {
            background: #0f7938;
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #64748b;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #475569;
        }
        
        .info-box {
            background: #ecf9f1;
            border: 1px solid #6ee7b7;
            padding: 15px;
            border-radius: 8px;
            margin-top: 10px;
            color: #065f46;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🖼️ Hero Slider Image Upload - Verification Test</h1>
            <p>Check if image upload functionality is properly configured</p>
        </div>
        
        <div class="content">
            <?php if (!$isLoggedIn): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i>
                    <strong>Authentication Required</strong> - Please <a href="login.php">login</a> to access this test
                </div>
            <?php elseif (!$isAdmin): ?>
                <div class="alert alert-warning">
                    <i class="fas fa-lock"></i>
                    <strong>Admin Access Required</strong> - Only administrators can use the image upload feature
                </div>
            <?php else: ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <strong>Authentication OK</strong> - You are logged in as an admin
                </div>
            <?php endif; ?>
            
            <!-- 1. Database Setup -->
            <div class="test-section">
                <h2>📊 Database Configuration</h2>
                
                <?php
                // Check if hero_slider table exists
                $tableCheck = $conn->query("SHOW TABLES LIKE 'hero_slider'");
                $tableExists = $tableCheck && $tableCheck->num_rows > 0;
                
                // Check if slide_image column exists
                $columnCheck = $conn->query("SHOW COLUMNS FROM hero_slider LIKE 'slide_image'");
                $columnExists = $columnCheck && $columnCheck->num_rows > 0;
                
                // Get sample data
                $sampleData = null;
                if ($tableExists) {
                    $result = $conn->query("SELECT * FROM hero_slider LIMIT 1");
                    if ($result && $result->num_rows > 0) {
                        $sampleData = $result->fetch_assoc();
                    }
                }
                ?>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $tableExists ? 'success' : 'error'; ?>">
                        <?php echo $tableExists ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $tableExists ? 'hero_slider Table' : 'hero_slider Table - NOT FOUND'; ?></strong>
                        <small><?php echo $tableExists ? 'Database table exists and is accessible' : 'Table needs to be created'; ?></small>
                    </div>
                </div>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $columnExists ? 'success' : 'error'; ?>">
                        <?php echo $columnExists ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $columnExists ? 'slide_image Column' : 'slide_image Column - NOT FOUND'; ?></strong>
                        <small><?php echo $columnExists ? 'Column exists for storing image filenames' : 'Need to add column to database'; ?></small>
                    </div>
                </div>
                
                <?php if (!$tableExists || !$columnExists): ?>
                    <div class="info-box">
                        <strong>⚠️ Action Required:</strong> Run the SQL script to create/update the table:
                        <div class="code-block">
mysql -u root cleaning_service < create_hero_slider_table.sql
                        </div>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- 2. File System Setup -->
            <div class="test-section">
                <h2>📁 File System Configuration</h2>
                
                <?php
                $uploadDir = 'assets/uploads/hero-slider/';
                $uploadDirExists = is_dir($uploadDir);
                $uploadDirWritable = $uploadDirExists && is_writable($uploadDir);
                
                // Check for sample uploaded image
                $sampleImage = null;
                if ($uploadDirExists) {
                    $files = array_diff(scandir($uploadDir), ['.', '..', '.htaccess']);
                    if (!empty($files)) {
                        $sampleImage = reset($files);
                    }
                }
                ?>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $uploadDirExists ? 'success' : 'error'; ?>">
                        <?php echo $uploadDirExists ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $uploadDirExists ? 'Upload Directory Exists' : 'Upload Directory Missing'; ?></strong>
                        <small><?php echo $uploadDirExists ? 'Location: ' . $uploadDir : 'Need to create: ' . $uploadDir; ?></small>
                    </div>
                </div>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $uploadDirWritable ? 'success' : ($uploadDirExists ? 'warning' : 'error'); ?>">
                        <?php echo $uploadDirWritable ? '✓' : '⚠'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $uploadDirWritable ? 'Directory Writable' : 'Directory Not Writable'; ?></strong>
                        <small><?php echo $uploadDirWritable ? 'Can upload files to directory' : 'Check permissions (need 755 or 777)'; ?></small>
                    </div>
                </div>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $sampleImage ? 'success' : 'warning'; ?>">
                        <?php echo $sampleImage ? '✓' : '○'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $sampleImage ? 'Sample Image Found' : 'No Images Uploaded Yet'; ?></strong>
                        <small><?php echo $sampleImage ? 'Sample: ' . htmlspecialchars($sampleImage) : 'Upload your first image from admin panel'; ?></small>
                    </div>
                </div>
                
                <?php if (!$uploadDirExists || !$uploadDirWritable): ?>
                    <div class="info-box">
                        <strong>⚠️ Action Required:</strong> Set up upload directory:
                        <div class="code-block">
mkdir -p assets/uploads/hero-slider<br>
chmod 755 assets/uploads/hero-slider
                        </div>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- 3. Backend Setup -->
            <div class="test-section">
                <h2>⚙️ Backend Configuration</h2>
                
                <?php
                $handlerExists = file_exists('hero_slider_handler.php');
                $handlerContent = $handlerExists ? file_get_contents('hero_slider_handler.php') : '';
                $hasImageFunction = strpos($handlerContent, 'handleImageUpload') !== false;
                ?>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $handlerExists ? 'success' : 'error'; ?>">
                        <?php echo $handlerExists ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $handlerExists ? 'hero_slider_handler.php' : 'hero_slider_handler.php - MISSING'; ?></strong>
                        <small><?php echo $handlerExists ? 'Backend API file is present' : 'Create hero_slider_handler.php file'; ?></small>
                    </div>
                </div>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $hasImageFunction ? 'success' : 'error'; ?>">
                        <?php echo $hasImageFunction ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $hasImageFunction ? 'Image Handler Function' : 'Image Handler Function - MISSING'; ?></strong>
                        <small><?php echo $hasImageFunction ? 'handleImageUpload() function is implemented' : 'Need to add image handling code'; ?></small>
                    </div>
                </div>
            </div>
            
            <!-- 4. Frontend Setup -->
            <div class="test-section">
                <h2>🎨 Frontend Configuration</h2>
                
                <?php
                $adminExists = file_exists('admin-dashboard.php');
                $adminContent = $adminExists ? file_get_contents('admin-dashboard.php') : '';
                $hasUploadSection = strpos($adminContent, 'image-upload-section') !== false;
                $hasImageHandler = strpos($adminContent, 'handleImageUpload') !== false;
                ?>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $adminExists ? 'success' : 'error'; ?>">
                        <?php echo $adminExists ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $adminExists ? 'admin-dashboard.php' : 'admin-dashboard.php - MISSING'; ?></strong>
                        <small><?php echo $adminExists ? 'Admin panel file is present' : 'Create admin-dashboard.php file'; ?></small>
                    </div>
                </div>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $hasUploadSection ? 'success' : 'error'; ?>">
                        <?php echo $hasUploadSection ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $hasUploadSection ? 'Upload Form Section' : 'Upload Form Section - MISSING'; ?></strong>
                        <small><?php echo $hasUploadSection ? 'Image upload form is in modal' : 'Need to add upload form HTML'; ?></small>
                    </div>
                </div>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $hasImageHandler ? 'success' : 'error'; ?>">
                        <?php echo $hasImageHandler ? '✓' : '✗'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $hasImageHandler ? 'JavaScript Image Handlers' : 'JavaScript Image Handlers - MISSING'; ?></strong>
                        <small><?php echo $hasImageHandler ? 'Image preview and upload JS is implemented' : 'Need to add JavaScript handlers'; ?></small>
                    </div>
                </div>
            </div>
            
            <!-- 5. Quick Tests -->
            <div class="test-section">
                <h2>🧪 Quick Functionality Tests</h2>
                
                <?php
                $allPassed = $tableExists && $columnExists && $uploadDirExists && $uploadDirWritable && 
                           $handlerExists && $hasImageFunction && $adminExists && $hasUploadSection && $hasImageHandler;
                ?>
                
                <div class="test-item">
                    <div class="status-icon <?php echo $allPassed ? 'success' : 'warning'; ?>">
                        <?php echo $allPassed ? '✓' : '⚠'; ?>
                    </div>
                    <div class="status-text">
                        <strong><?php echo $allPassed ? 'All Components Ready' : 'Some Components Need Setup'; ?></strong>
                        <small><?php echo $allPassed ? 'Image upload feature is fully configured and ready to use' : 'Fix the issues above to enable image uploads'; ?></small>
                    </div>
                </div>
            </div>
            
            <!-- 6. Next Steps -->
            <div class="test-section">
                <h2>🚀 Next Steps</h2>
                
                <?php if ($allPassed): ?>
                    <div class="info-box" style="background: #d1fae5; border-color: #6ee7b7; color: #065f46;">
                        <strong>✓ Ready to Go!</strong><br>
                        1. Go to Admin Dashboard → Hero Slider<br>
                        2. Click "Add Slide"<br>
                        3. Scroll to "Slide Image" section<br>
                        4. Upload or drag-drop an image<br>
                        5. Click "Save Slide"<br>
                        6. Image will appear on homepage hero slider
                    </div>
                <?php else: ?>
                    <div class="info-box">
                        <strong>⚠️ Setup Required</strong><br>
                        Please fix the issues highlighted above:
                        <ul style="margin-left: 20px; margin-top: 10px;">
                            <?php if (!$tableExists || !$columnExists): ?>
                                <li>Create database table: Run create_hero_slider_table.sql</li>
                            <?php endif; ?>
                            <?php if (!$uploadDirExists): ?>
                                <li>Create upload directory: assets/uploads/hero-slider/</li>
                            <?php endif; ?>
                            <?php if (!$uploadDirWritable): ?>
                                <li>Make directory writable (chmod 755)</li>
                            <?php endif; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- 7. Actions -->
            <div class="button-group">
                <?php if ($isAdmin && $allPassed): ?>
                    <a href="admin-dashboard.php" class="btn btn-primary">Go to Admin Dashboard</a>
                <?php endif; ?>
                <a href="javascript:location.reload()" class="btn btn-secondary">Refresh Test</a>
            </div>
        </div>
    </div>
</body>
</html>
